/*****************************************************************************
 * postproc.c: video postprocessing using libpostproc
 *****************************************************************************
 * Copyright (C) 1999-2009 VLC authors and VideoLAN
 *
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 *          Gildas Bazin <gbazin@netcourrier.com>
 *          Antoine Cellerier <dionoea at videolan dot org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

/*****************************************************************************
 * NOTA BENE: this module requires the linking against a library which is
 * known to require licensing under the GNU General Public License version 2
 * (or later). Therefore, the result of compiling this module will normally
 * be subject to the terms of that later license.
 *****************************************************************************/


#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <vlc_common.h>
#include <vlc_configuration.h>
#include <vlc_plugin.h>
#include <vlc_filter.h>
#include <vlc_picture.h>
#include <vlc_cpu.h>

#ifdef HAVE_POSTPROC_POSTPROCESS_H
#   include <postproc/postprocess.h>
#else
#   include <libpostproc/postprocess.h>
#endif

#ifndef PP_CPU_CAPS_AUTO
#   define PP_CPU_CAPS_AUTO 0
#endif

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/
static int OpenPostproc( filter_t * );

VIDEO_FILTER_WRAPPER_CLOSE(PostprocPict, ClosePostproc)

static int PPQCallback( vlc_object_t *, char const *,
                        vlc_value_t, vlc_value_t, void * );
static int PPNameCallback( vlc_object_t *, char const *,
                           vlc_value_t, vlc_value_t, void * );

#define Q_TEXT N_("Post processing quality")
#define Q_LONGTEXT N_( \
    "Quality of post processing. Valid range is 0 (disabled) to 6 (highest)\n"     \
    "Higher levels require more CPU power, but produce higher quality pictures.\n" \
    "With default filter chain, the values map to the following filters:\n"        \
    "1: hb, 2-4: hb+vb, 5-6: hb+vb+dr" )

#define NAME_TEXT N_("FFmpeg post processing filter chains")

#define FILTER_PREFIX "postproc-"

/*****************************************************************************
 * Module descriptor
 *****************************************************************************/
vlc_module_begin ()
    set_description( N_("Video post processing filter") )
    set_shortname( N_("Postproc" ) )
    add_shortcut( "postprocess", "pp" ) /* name is "postproc" */
    set_subcategory( SUBCAT_VIDEO_VFILTER )

    set_callback_video_filter( OpenPostproc )

    add_integer_with_range( FILTER_PREFIX "q", PP_QUALITY_MAX, 0,
                            PP_QUALITY_MAX, Q_TEXT, Q_LONGTEXT )
        change_safe()
    add_string( FILTER_PREFIX "name", "default", NAME_TEXT,
                NULL )
vlc_module_end ()

static const char *const ppsz_filter_options[] = {
    "q", "name", NULL
};

/*****************************************************************************
 * filter_sys_t : libpostproc video postprocessing descriptor
 *****************************************************************************/
typedef struct
{
    /* Never changes after init */
    pp_context *pp_context;

    /* Set to NULL if post processing is disabled */
    pp_mode *pp_mode;

    /* Lock when using or changing pp_mode */
    vlc_mutex_t lock;
} filter_sys_t;


/*****************************************************************************
 * OpenPostproc: probe and open the postproc
 *****************************************************************************/
static int OpenPostproc( filter_t *p_filter )
{
    filter_sys_t *p_sys;
    vlc_value_t val, val_orig;
    const char *desc;
    int i_flags = PP_CPU_CAPS_AUTO;

    if( !video_format_IsSameChroma( &p_filter->fmt_in.video,
                                    &p_filter->fmt_out.video ) ||
        p_filter->fmt_in.video.i_height != p_filter->fmt_out.video.i_height ||
        p_filter->fmt_in.video.i_width != p_filter->fmt_out.video.i_width )
    {
        msg_Err( p_filter, "Filter input and output formats must be identical" );
        return VLC_EGENERIC;
    }

    switch( p_filter->fmt_in.video.i_chroma )
    {
        case VLC_CODEC_I444:
        /* case VLC_CODEC_YUVA:
           FIXME: Should work but alpha plane needs to be copied manually and
                  I'm kind of feeling too lazy to write the code to do that ATM
                  (i_pitch vs i_visible_pitch...). */
            i_flags |= PP_FORMAT_444;
            break;
        case VLC_CODEC_I422:
            i_flags |= PP_FORMAT_422;
            break;
        case VLC_CODEC_I411:
            i_flags |= PP_FORMAT_411;
            break;
        case VLC_CODEC_I420:
        case VLC_CODEC_YV12:
            i_flags |= PP_FORMAT_420;
            break;
        default:
            msg_Err( p_filter, "Unsupported input chroma (%4.4s)",
                      (char*)&p_filter->fmt_in.video.i_chroma );
            return VLC_EGENERIC;
    }

    p_sys = malloc( sizeof( filter_sys_t ) );
    if( !p_sys ) return VLC_ENOMEM;
    p_filter->p_sys = p_sys;

    p_sys->pp_context = pp_get_context( p_filter->fmt_in.video.i_width,
                                        p_filter->fmt_in.video.i_height,
                                        i_flags );
    if( !p_sys->pp_context )
    {
        msg_Err( p_filter, "Error while creating post processing context." );
        free( p_sys );
        return VLC_EGENERIC;
    }

    config_ChainParse( p_filter, FILTER_PREFIX, ppsz_filter_options,
                       p_filter->p_cfg );

    var_Create( p_filter, FILTER_PREFIX "q", VLC_VAR_INTEGER |
                VLC_VAR_DOINHERIT | VLC_VAR_ISCOMMAND );
    var_Change( p_filter, FILTER_PREFIX "q", VLC_VAR_SETTEXT,
                _("Post processing") );

    var_Get( p_filter, FILTER_PREFIX "q", &val_orig );
    var_Change( p_filter, FILTER_PREFIX "q", VLC_VAR_DELCHOICE, val_orig );

    val.psz_string = var_GetNonEmptyString( p_filter, FILTER_PREFIX "name" );
    if( val_orig.i_int )
    {
        p_sys->pp_mode = pp_get_mode_by_name_and_quality( val.psz_string ?
                                                          val.psz_string :
                                                          "default",
                                                          val_orig.i_int );

        if( !p_sys->pp_mode )
        {
            msg_Err( p_filter, "Error while creating post processing mode." );
            free( val.psz_string );
            pp_free_context( p_sys->pp_context );
            free( p_sys );
            return VLC_EGENERIC;
        }
    }
    else
    {
        p_sys->pp_mode = NULL;
    }
    free( val.psz_string );

    for( val.i_int = 0; val.i_int <= PP_QUALITY_MAX; val.i_int++ )
    {
        switch( val.i_int )
        {
            case 0:
                desc = _("Disable");
                break;
            case 1:
                desc = _("Lowest");
                break;
            case PP_QUALITY_MAX:
                desc = _("Highest");
                break;
            default:
                desc = NULL;
                break;
        }
        var_Change( p_filter, FILTER_PREFIX "q", VLC_VAR_ADDCHOICE, val, desc );
    }

    vlc_mutex_init( &p_sys->lock );

    /* Add the callback at the end to prevent crashes */
    var_AddCallback( p_filter, FILTER_PREFIX "q", PPQCallback, NULL );
    var_AddCallback( p_filter, FILTER_PREFIX "name", PPNameCallback, NULL );

    p_filter->ops = &PostprocPict_ops;

    msg_Warn( p_filter, "Quantification table was not set by video decoder. "
                        "Postprocessing won't look good." );
    return VLC_SUCCESS;
}

/*****************************************************************************
 * ClosePostproc
 *****************************************************************************/
static void ClosePostproc( filter_t *p_filter )
{
    filter_sys_t *p_sys = p_filter->p_sys;

    /* delete the callback before destroying the mutex */
    var_DelCallback( p_filter, FILTER_PREFIX "q", PPQCallback, NULL );
    var_DelCallback( p_filter, FILTER_PREFIX "name", PPNameCallback, NULL );

    /* Destroy the resources */
    pp_free_context( p_sys->pp_context );
    pp_free_mode( p_sys->pp_mode );
    free( p_sys );
}

/*****************************************************************************
 * PostprocPict
 *****************************************************************************/
static void PostprocPict( filter_t *p_filter, picture_t *p_pic, picture_t *p_outpic )
{
    filter_sys_t *p_sys = p_filter->p_sys;

    /* Lock to prevent issues if pp_mode is changed */
    vlc_mutex_lock( &p_sys->lock );
    if( p_sys->pp_mode != NULL )
    {
        const uint8_t *src[3];
        uint8_t *dst[3];
        int i_src_stride[3], i_dst_stride[3];

        for( int i_plane = 0; i_plane < p_pic->i_planes; i_plane++ )
        {
            src[i_plane] = p_pic->p[i_plane].p_pixels;
            dst[i_plane] = p_outpic->p[i_plane].p_pixels;

            /* I'm not sure what happens if i_pitch != i_visible_pitch ...
             * at least it shouldn't crash. */
            i_src_stride[i_plane] = p_pic->p[i_plane].i_pitch;
            i_dst_stride[i_plane] = p_outpic->p[i_plane].i_pitch;
        }

        pp_postprocess( src, i_src_stride, dst, i_dst_stride,
                        p_filter->fmt_in.video.i_width,
                        p_filter->fmt_in.video.i_height, NULL, 0,
                        p_sys->pp_mode, p_sys->pp_context, 0 );
    }
    else
        picture_CopyPixels( p_outpic, p_pic );
    vlc_mutex_unlock( &p_sys->lock );
}

/*****************************************************************************
 * PPChangeMode: change the current mode and quality
 *****************************************************************************/
static void PPChangeMode( filter_t *p_filter, const char *psz_name,
                          int i_quality )
{
    filter_sys_t *p_sys = p_filter->p_sys;
    pp_mode *newmode = NULL, *oldmode;

    if( i_quality > 0 )
    {
         newmode = pp_get_mode_by_name_and_quality( psz_name ? psz_name :
                                                    "default", i_quality );
         if( newmode == NULL )
         {
             msg_Warn( p_filter, "Error while changing post processing mode. "
                       "Keeping previous mode." );
             return;
        }
    }

    vlc_mutex_lock( &p_sys->lock );
    oldmode = p_sys->pp_mode;
    p_sys->pp_mode = newmode;
    vlc_mutex_unlock( &p_sys->lock );

    pp_free_mode( oldmode );
}

static int PPQCallback( vlc_object_t *p_this, const char *psz_var,
                        vlc_value_t oldval, vlc_value_t newval, void *p_data )
{
    VLC_UNUSED(psz_var); VLC_UNUSED(oldval); VLC_UNUSED(p_data);
    filter_t *p_filter = (filter_t *)p_this;

    char *psz_name = var_GetNonEmptyString( p_filter, FILTER_PREFIX "name" );
    PPChangeMode( p_filter, psz_name, newval.i_int );
    free( psz_name );
    return VLC_SUCCESS;
}

static int PPNameCallback( vlc_object_t *p_this, const char *psz_var,
                           vlc_value_t oldval, vlc_value_t newval, void *p_data )
{
    VLC_UNUSED(psz_var); VLC_UNUSED(oldval); VLC_UNUSED(p_data);
    filter_t *p_filter = (filter_t *)p_this;

    int i_quality = var_GetInteger( p_filter, FILTER_PREFIX "q" );
    PPChangeMode( p_filter, *newval.psz_string ? newval.psz_string : NULL,
                  i_quality );
    return VLC_SUCCESS;
}
