// Copyright (C) 2025 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#include "qquickcoloropacityanimation_p.h"

#include <QtQuick/private/qquickanimation_p_p.h>

QT_BEGIN_NAMESPACE

/*!
    \qmlmodule QtQuick.VectorImage.Helpers
    \title Qt Quick Vector Image Helpers QML Types
    \ingroup qmlmodules
    \brief Provides QML types used by VectorImage and related tools.
    \since 6.10

    This module contains types used in scenes generated by \l{VectorImage}, \l{svgtoqml} and related
    tools. The types are made to replicate specialized behavior defined by the vector graphics file
    formats it loads and are not intended to be generally useful.

    To use the types in this module, import the module with the following line:

    \qml
    import QtQuick.VectorImage.Helpers
    \endqml

    \section1 QML Types
*/

static QVariant opacityInterpolator(const QColor &from,
                                    const QColor &to,
                                    const QVariant &current,
                                    qreal progress)
{
    QColor color = current.value<QColor>();

    qreal fromAlpha = from.alphaF();
    qreal toAlpha = to.alphaF();

    color.setAlphaF(fromAlpha + (toAlpha - fromAlpha) * progress);
    return QVariant::fromValue(color);
}

/*!
    \qmltype ColorOpacityAnimation
    \inqmlmodule QtQuick.VectorImage.Helpers
    \inherits Item
    \brief Animates the alpha value of a color without modifying the rest of the color.

    This type will animate the alpha value of a color without modifying the other components. It
    is used to support the \c fill-opacity and \c stroke-opacity properties of SVG, which can be
    animated separately from the opacity of the shape itself.
*/
QQuickColorOpacityAnimation::QQuickColorOpacityAnimation(QObject *parent)
    : QQuickPropertyAnimation(parent)
{
    Q_D(QQuickPropertyAnimation);
    d->defaultToInterpolatorType = true;
    d->interpolatorType = QMetaType::QColor;
    d->extendedInterpolator = reinterpret_cast<QQuickPropertyAnimationPrivate::ExtendedInterpolator>(reinterpret_cast<void(*)()>(opacityInterpolator));
}

/*!
    \qmlproperty real QtQuick.VectorImage.Helpers::ColorOpacityAnimation::from

    The value of the color's alpha channel at the start of the animation.
*/
qreal QQuickColorOpacityAnimation::from() const
{
    Q_D(const QQuickPropertyAnimation);
    return d->from.value<QColor>().alphaF();
}

void QQuickColorOpacityAnimation::setFrom(qreal from)
{
    QColor color = Qt::red;
    color.setAlphaF(from);

    QQuickPropertyAnimation::setFrom(color);
}

/*!
    \qmlproperty real QtQuick.VectorImage.Helpers::ColorOpacityAnimation::to

    The value of the color's alpha channel at the end of the animation.
*/
qreal QQuickColorOpacityAnimation::to() const
{
    Q_D(const QQuickPropertyAnimation);
    return d->to.value<QColor>().alphaF();
}

void QQuickColorOpacityAnimation::setTo(qreal to)
{
    QColor color = Qt::yellow;
    color.setAlphaF(to);

    QQuickPropertyAnimation::setTo(color);
}

QT_END_NAMESPACE

#include <moc_qquickcoloropacityanimation_p.cpp>
