#!/bin/sh
# -*- mode: shell-script; indent-tabs-mode: nil; sh-basic-offset: 4; -*-
# ex: ts=8 sw=4 sts=4 et filetype=sh

type getarg >/dev/null 2>&1 || . /lib/dracut-lib.sh

# Get boot parameters
kdeosisolabel=$(getarg kdeosisolabel=)
kdeosisobasedir=$(getarg kdeosisobasedir=)
kdeosisodevice=$(getarg kdeosisodevice=)

# Set defaults
[ -z "$kdeosisobasedir" ] && kdeosisobasedir="kdeos"
[ -z "$kdeosisodevice" ] && kdeosisodevice="/dev/disk/by-label/${kdeosisolabel}"

# Architecture
arch=$(uname -m)

# Directories
rw_branch_size="75%"

info "KaOS Live: Starting live mount"

# Wait for device
if [ -n "$kdeosisodevice" ]; then
    info "KaOS Live: Waiting for device $kdeosisodevice"
    
    # Wait for device to appear
    count=0
    while [ ! -b "$kdeosisodevice" ] && [ $count -lt 30 ]; do
        sleep 1
        count=$((count + 1))
    done
    
    if [ ! -b "$kdeosisodevice" ]; then
        warn "KaOS Live: Device $kdeosisodevice not found after 30 seconds"
        return 1
    fi
    
    # Mount the ISO
    mkdir -p /bootmnt
    mount -r "$kdeosisodevice" /bootmnt || {
        warn "KaOS Live: Failed to mount $kdeosisodevice"
        return 1
    }
    
    info "KaOS Live: Mounted $kdeosisodevice to /bootmnt"
fi

# Check for isomounts file
isomounts="/bootmnt/${kdeosisobasedir}/isomounts"
if [ ! -f "$isomounts" ]; then
    warn "KaOS Live: isomounts file not found at $isomounts"
    return 1
fi

# Create rw branch
info "KaOS Live: Creating rw_branch (tmpfs)"
mkdir -p /rw_branch
mount -t tmpfs -o "size=${rw_branch_size}",mode=0755 rw_branch /rw_branch

# Mount all squashfs images and build layer list
info "KaOS Live: Mounting squashfs images"
layer_list=""

while read img imgarch mountpoint type; do
    # Skip comments
    [ "${img#\#}" != "${img}" ] && continue
    
    # Check architecture
    [ "$imgarch" != "$arch" ] && continue
    
    # Check if file exists
    [ ! -r "/bootmnt/${kdeosisobasedir}/${img}" ] && continue
    
    if [ "${type}" = "squashfs" ]; then
        img_fullname="${img##*/}"
        img_name="${img_fullname%.*}"
        tmp_mnt="/ro_branch/${img_name}"
        
        info "KaOS Live: Mounting ${img_fullname}"
        
        mkdir -p "${tmp_mnt}"
        mount -r -t squashfs "/bootmnt/${kdeosisobasedir}/${img}" "${tmp_mnt}" || {
            warn "KaOS Live: Failed to mount ${img}"
            continue
        }
        
        # Build layer list (prepend for correct OverlayFS priority)
        if [ -z "${layer_list}" ]; then
            layer_list="${tmp_mnt}"
        else
            layer_list="${tmp_mnt}:${layer_list}"
        fi
    fi
done < "$isomounts"

if [ -z "${layer_list}" ]; then
    warn "KaOS Live: No squashfs images were mounted"
    return 1
fi

# Create overlay
info "KaOS Live: Creating overlay filesystem"
info "KaOS Live: Layers: ${layer_list}"

mkdir -p /rw_branch/.work

mount -t overlay -o lowerdir="${layer_list}",upperdir=/rw_branch,workdir=/rw_branch/.work overlay "$NEWROOT" || {
    warn "KaOS Live: Failed to create overlay"
    return 1
}

info "KaOS Live: Overlay mounted successfully"

# Bind mount /bootmnt to newroot so it's available after switch_root
mkdir -p "$NEWROOT/bootmnt"
mount -o bind /bootmnt "$NEWROOT/bootmnt"

info "KaOS Live: Live mount complete"

return 0
