"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.canPossiblyPreviewExtension = canPossiblyPreviewExtension;
exports.displayQuickPreviewWindow = displayQuickPreviewWindow;
exports.generatePreview = generatePreview;
const child_process_1 = require("child_process");
const path_1 = __importDefault(require("path"));
const electron_1 = require("electron");
async function generatePreviewToken(previewPath) {
    return electron_1.ipcRenderer.invoke('quickpreview:generateToken', previewPath);
}
function cleanupPreviewToken(token) {
    electron_1.ipcRenderer.invoke('quickpreview:cleanupToken', token);
}
const QuickPreviewCSP = [
    "default-src 'self'",
    "script-src 'self' 'unsafe-inline'",
    "style-src 'self' 'unsafe-inline'",
    "img-src 'self' data: https: http:",
    "object-src 'none'",
    "frame-src 'none'",
    "base-uri 'self'",
].join('; ');
let quickPreviewWindow = null;
let captureWindow = null;
const captureQueue = [];
const filesRoot = __dirname.replace('app.asar', 'app.asar.unpacked');
const FileSizeLimit = 5 * 1024 * 1024;
const ThumbnailWidth = 320 * (11 / 8.5);
const QuicklookIsAvailable = process.platform === 'darwin';
const PDFJSRoot = path_1.default.join(filesRoot, 'pdfjs-4.3.136');
const QuicklookBlacklist = [
    'jpg',
    'bmp',
    'gif',
    'png',
    'jpeg',
    'zip',
    'tar',
    'gz',
    'bz2',
    'dmg',
    'exe',
    'ics',
];
const CrossplatformStrategies = {
    pdfjs: ['pdf'],
    mammoth: ['docx'],
    snarkdown: ['md'],
    xlsx: ['xls', 'xlsx', 'csv', 'eth', 'ods', 'fods', 'uos1', 'uos2', 'dbf', 'prn', 'xlw', 'xlsb'],
    prism: [
        'html',
        'svg',
        'xml',
        'css',
        'c',
        'cc',
        'cpp',
        'js',
        'jsx',
        'tsx',
        'ts',
        'go',
        'cs',
        'patch',
        'swift',
        'java',
        'json',
        'jsonp',
        'tex',
        'mm',
        'm',
        'h',
        'py',
        'rb',
        'rs',
        'sql',
        'yml',
        'yaml',
        'txt',
        'log',
    ],
};
const CrossplatformStrategiesBetterThanQuicklook = ['snarkdown', 'prism'];
function strategyForPreviewing(ext) {
    if (ext.startsWith('.'))
        ext = ext.substr(1);
    const strategy = Object.keys(CrossplatformStrategies).find(strategy => CrossplatformStrategies[strategy].includes(ext));
    if (QuicklookIsAvailable && !QuicklookBlacklist.includes(ext)) {
        if (!strategy || !CrossplatformStrategiesBetterThanQuicklook.includes(strategy)) {
            return 'quicklook';
        }
    }
    return strategy;
}
const PreviewWindowMenuTemplate = [
    {
        label: 'File',
        role: 'window',
        submenu: [
            {
                label: 'Minimize Window',
                accelerator: 'CmdOrCtrl+M',
                role: 'minimize',
            },
            {
                label: 'Close Window',
                accelerator: 'CmdOrCtrl+W',
                role: 'close',
            },
        ],
    },
    {
        label: 'Edit',
        submenu: [
            {
                label: 'Undo',
                accelerator: 'CmdOrCtrl+Z',
                role: 'undo',
            },
            {
                label: 'Redo',
                accelerator: 'Shift+CmdOrCtrl+Z',
                role: 'redo',
            },
            {
                type: 'separator',
            },
            {
                label: 'Cut',
                accelerator: 'CmdOrCtrl+X',
                role: 'cut',
            },
            {
                label: 'Copy',
                accelerator: 'CmdOrCtrl+C',
                role: 'copy',
            },
            {
                label: 'Paste',
                accelerator: 'CmdOrCtrl+V',
                role: 'paste',
            },
            {
                label: 'Select All',
                accelerator: 'CmdOrCtrl+A',
                role: 'selectAll',
            },
        ],
    },
    {
        label: 'View',
        submenu: [
            {
                label: 'Reload',
                accelerator: 'CmdOrCtrl+R',
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.reload();
                },
            },
            {
                label: 'Toggle Full Screen',
                accelerator: (function () {
                    if (process.platform === 'darwin')
                        return 'Ctrl+Command+F';
                    else
                        return 'F11';
                })(),
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.setFullScreen(!focusedWindow.isFullScreen());
                },
            },
            {
                label: 'Toggle Developer Tools',
                accelerator: (function () {
                    if (process.platform === 'darwin')
                        return 'Alt+Command+I';
                    else
                        return 'Ctrl+Shift+I';
                })(),
                click: function (item, focusedWindow) {
                    if (focusedWindow)
                        focusedWindow.webContents.toggleDevTools();
                },
            },
        ],
    },
];
function canPossiblyPreviewExtension(file) {
    if (file.size > FileSizeLimit) {
        return false;
    }
    return !!strategyForPreviewing(file.displayExtension());
}
function displayQuickPreviewWindow(filePath) {
    const isPDF = filePath.endsWith('.pdf');
    const strategy = strategyForPreviewing(path_1.default.extname(filePath));
    if (strategy === 'quicklook') {
        const currentWin = AppEnv.getCurrentWindow();
        currentWin.previewFile(filePath);
        return;
    }
    if (quickPreviewWindow === null) {
        const { BrowserWindow } = require('@electron/remote');
        quickPreviewWindow = new BrowserWindow({
            width: 800,
            height: 600,
            center: true,
            skipTaskbar: true,
            backgroundColor: isPDF ? '#404040' : '#FFF',
            webPreferences: {
                preload: path_1.default.join(filesRoot, 'preload.js'),
                nodeIntegration: false,
                contextIsolation: true,
            },
        });
        quickPreviewWindow.webContents.session.webRequest.onHeadersReceived((details, callback) => {
            callback({
                responseHeaders: Object.assign(Object.assign({}, details.responseHeaders), { 'Content-Security-Policy': [QuickPreviewCSP] }),
            });
        });
        quickPreviewWindow.once('closed', () => {
            quickPreviewWindow = null;
        });
        quickPreviewWindow.setMenu(require('@electron/remote').Menu.buildFromTemplate(PreviewWindowMenuTemplate));
    }
    else {
        quickPreviewWindow.show();
    }
    quickPreviewWindow.setTitle(path_1.default.basename(filePath));
    if (isPDF) {
        quickPreviewWindow.loadFile(path_1.default.join(PDFJSRoot, 'web/viewer.html'), {
            search: `file=${encodeURIComponent(`file://${filePath}`)}`,
        });
    }
    else {
        quickPreviewWindow.loadFile(path_1.default.join(filesRoot, 'renderer.html'), {
            search: JSON.stringify({ mode: 'display', filePath, strategy }),
        });
    }
}
async function generatePreview({ file, filePath, previewPath, }) {
    const strategy = strategyForPreviewing(file.displayExtension());
    if (strategy === 'quicklook') {
        return await _generateQuicklookPreview({ filePath });
    }
    else if (strategy) {
        return await _generateCrossplatformPreview({ file, filePath, previewPath, strategy });
    }
    else {
        return false;
    }
}
async function _generateCrossplatformPreview({ file, filePath, previewPath, strategy }) {
    return new Promise(resolve => {
        captureQueue.push({ file, filePath, previewPath, strategy, resolve });
        if (!captureWindow || captureWindow.isDestroyed()) {
            captureWindow = _createCaptureWindow();
            _generateNextCrossplatformPreview();
        }
    });
}
function _createCaptureWindow() {
    const { BrowserWindow } = require('@electron/remote');
    const win = new BrowserWindow({
        width: ThumbnailWidth,
        height: ThumbnailWidth,
        show: false,
        webPreferences: {
            preload: path_1.default.join(filesRoot, 'preload.js'),
            nodeIntegration: false,
            contextIsolation: true,
        },
    });
    win.webContents.session.webRequest.onHeadersReceived((details, callback) => {
        callback({
            responseHeaders: Object.assign(Object.assign({}, details.responseHeaders), { 'Content-Security-Policy': [QuickPreviewCSP] }),
        });
    });
    win.webContents.on('render-process-gone', (event, details) => {
        console.warn(`Thumbnail generation webcontents crashed (reason: ${details.reason}).`);
        if (captureWindow === win)
            captureWindow = null;
        win.destroy();
    });
    win.once('closed', () => {
        if (captureWindow === win)
            captureWindow = null;
    });
    return win;
}
async function _generateNextCrossplatformPreview() {
    if (captureQueue.length === 0) {
        if (captureWindow && !captureWindow.isDestroyed()) {
            captureWindow.destroy();
        }
        else {
            console.warn(`Thumbnail generation finished but window is already destroyed.`);
        }
        captureWindow = null;
        return;
    }
    const { strategy, filePath, previewPath, resolve } = captureQueue.pop();
    const previewToken = await generatePreviewToken(previewPath);
    captureWindow.loadFile(path_1.default.join(filesRoot, 'renderer.html'), {
        search: JSON.stringify({ strategy, mode: 'capture', filePath, previewToken }),
    });
    let onFinalize = null;
    const timer = setTimeout(() => {
        console.warn(`Thumbnail generation timed out for ${filePath}`);
        onFinalize(false);
    }, 5000);
    const onRendererSuccess = () => {
        onFinalize(true);
    };
    onFinalize = success => {
        clearTimeout(timer);
        if (captureWindow) {
            captureWindow.removeListener('page-title-updated', onRendererSuccess);
        }
        if (!success) {
            cleanupPreviewToken(previewToken);
        }
        process.nextTick(_generateNextCrossplatformPreview);
        resolve(success);
    };
    captureWindow.once('page-title-updated', onRendererSuccess);
}
async function _generateQuicklookPreview({ filePath }) {
    const dirQuoted = path_1.default.dirname(filePath).replace(/"/g, '\\"');
    const pathQuoted = filePath.replace(/"/g, '\\"');
    return new Promise(resolve => {
        const cmd = '/usr/bin/qlmanage';
        const args = [
            '-t',
            '-f',
            `${window.devicePixelRatio}`,
            '-s',
            `${ThumbnailWidth}`,
            '-o',
            dirQuoted,
            pathQuoted,
        ];
        (0, child_process_1.execFile)(cmd, args, (error, stdout, stderr) => {
            if (error ||
                stdout.match(/No thumbnail created/i) ||
                (stderr && !stdout.includes('produced one thumbnail'))) {
                resolve(false);
            }
            else {
                resolve(true);
            }
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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